/*******************************************************************************

    Filename: Fiero_GT_ALDL_Adapter_V1_6_0.ino
      Author: Paul Romsky
     Company: Xyaxis - Romsky Consultation Consortium
    Document: 008-00001-901
        Date: 01 AUG 2022
     Project: Fiero GT ALDL Monitor
   Processor: Arduino UNO R3 or Adafruit Trinket M0
 Description: This is the microcontroller code to monitor and emulate the
              Fiero GT ALDL 160 Baud Comm Link and the Service Engine Soon (SES)
              signal - if the Fiero is modified to include the SES signal on 
              the Assembly Line Diagnostic Link (ALDL) connector. This code is 
              to be installed on the Fiero GT ALDL Adapter V1.6.0 that interfaces 
              to the Fiero GT ALDL Monitor V1.6.0 - a Graphical User Interface (GUI)
              designed for this adapter.  The adapter sends interpreted signals 
              from the Fiero to the GUI installed on a Personal Computer (PC) 
              via a USB serial Comm link at 9600 Baud, 8 Bits, No Parity, 1 Stop.

 Program via the Arduino IDE:
 Note: Be sure to use the Up and Down arrows on the pulldown fields as there may be 
 many more selections than are shown.
 
              Target: Ardunio UNO ($23 USD Board)
               Board: Arduino Uno
           USB Stack: Arduino
            Optimize: Small (-Os) (standard)
          Programmer: ArduinoISP
 Compile Time Switch: #define TARGET_BOARD (TARGET_BOARD_ARDUINO_UNO_R3)
 
              Target: Adafruit Trinket M0 ($9 USD Board)
               Board: Adafruit Trinket M0 (SAMD 21)
           USB Stack: Arduino
            Optimize: Small (-Os) (standard)
          Programmer: USBtinyISP
 Compile Time Switch: #define TARGET_BOARD (TARGET_BOARD_ADAFRUIT_TRINKET_M0)

 For reliable uploads to the Adafruit Trinket M0: Before uploading, on the Trinket M0,
 press the Reset button twice within a half second.  This will put the Trinket M0 in 
 Bootloader mode (the Red LED will flash slowly between bright/dim and the Dotstar LED will be bright Green).
 After uploading and the code is running on the Trinket M0, the Dotstar LED will be normally bright 
 Magneta (by default) - ignore any color of the Dotstar LED as it is not controlled by this code.
 
 The Trinket M0 comes delivered with Circuit Python installed and acts as a mini thumb drive when plugged
 into the PC.  Once this code is installed on the Trinket M0, Circuit Python will be removed from the 
 Trinket M0 and it will not longer appear as a mini thumb drive when plugged into the PC. 
 If you want to later reuse the Trinket M0 for other applications and revert the Trinket M0 back to 
 Circuit Phyton, refer to the Adafruit website for details.  Do not port this code to Circuit Python 
 for the Trinket M0, Circuit Pyhton is too slow and does not have the microseconds counter function micros().

 Copyright (c) Paul Romsky 2022 - All rights reserved
  
 Revision History:
 
 Rev    Date        Name                   Description
 ------ ----------- ---------------------- ----------------------------------
  0.0.0 01 AUG 2022 Paul Romsky            Initial coding.

*******************************************************************************/

/* Compile Time Switches */

#define TARGET_BOARD_ARDUINO_UNO_R3      1              /* Target the Arduino UNO R3 board */
#define TARGET_BOARD_ADAFRUIT_TRINKET_M0 1              /* Target the Adafruit Trinket M0 board */
//#define TARGET_BOARD (TARGET_BOARD_ARDUINO_UNO_R3)      /* Compile the code for Target board Selected (Comment out if not the board desired) */
#define TARGET_BOARD (TARGET_BOARD_ADAFRUIT_TRINKET_M0) /* Compile the code for Target board Selected (Comment out if not the board desired) */

#define SIMULATE_DEFAULT  0 /* 0 = Normal, 1 = Turn on Simulation by Default */


/* Constants */

#if TARGET_BOARD == TARGET_BOARD_ARDUINO_UNO_R3

#define TRANSISTOR_PWM_OFF                0 /* A Constant Low Pulse Width Moulation (PWM) to Transistor Input Low Pass Filter, Transitor is Off (Open between ALDL Connector DIAG Mode Select and Ground) */
#define TRANSISTOR_PWM_10K_OHMS          64 /* A Constant PWM to Transistor Input Low Pass Filter that sets Transitor partially On (10K Ohms between ALDL Connector DIAG Mode Select and Ground) */
#define TRANSISTOR_PWM_3_9K_OHMS        128 /* A Constant PWM to Transistor Input Low Pass Filter that sets Transitor partially On (3.9K Ohms between ALDL Connector DIAG Mode Select and Ground) */
#define TRANSISTOR_PWM_SATURATED        255 /* A Constant High PWM to Transistor Input Low Pass Filter, Transitor is Saturated (Short between ALDL Connector DIAG Mode Select and Ground) */

#define SES_ALIVE_COUNT               12200 /* The Service Engine Soon (SES) monitor will send an Error Code 0 (EC0) periodcially as it monitors the Error Codes to keep the Link active */ 

#define USER_LED_PIN                     13 /* Arduino UNO R3 Yellow 'L' LED */
#define ALDL_160_BAUD_PIN                 2 /* Arduino UNO R3 'DIGITAL – PWM~ (IOL)' Pin  2: Digtial Input from a 5.0V Zener limiter from the Fiero ALDL Connector 160 Baud Pin (ALDL Pin E) */
#define ALDL_SES_PIN                      4 /* Arduino UNO R3 'DIGITAL – PWM~ (IOL)' Pin  4: Digital Input from a 5.0V Zener limiter from the Fiero ALDL Connector SES Signal (ALDL Pin D) */
#define ALDL_DIAG_MODE_SELECT_PIN         3 /* Arduino UNO R3 'DIGITAL – PWM~ (IOL)' Pin ~3: PWM Output to a Transisitor to control the resistance to the Fiero ALDL Connector Diag Mode Select Pin (ALDL Pin B) */
                                            /* 
                                             *  The Arduino UNO R3 does not have a Digital to Analog Converter (DAC) output pin, so a PWM pin is used to generate continous PWM 
                                             *  pulses out of the pin, therefore, the transistor has a simple Resistor-Capacitor (RC) Low Pass Filter before its Base input to 
                                             *  convert the PWM stream to a steady analog DC voltage (effectively acting like a DAC).  
                                             */
                                             
#elif TARGET_BOARD == TARGET_BOARD_ADAFRUIT_TRINKET_M0

#define TRANSISTOR_PWM_OFF                0 /* A Constant DAC voltage to Transistor Input, Transitor is Off (Open between ALDL Connector DIAG Mode Select and Ground) */
#define TRANSISTOR_PWM_10K_OHMS         256 /* A Constant DAC voltage to Transistor Input, Transitor partially On (10K Ohms between ALDL Connector DIAG Mode Select and Ground) */
#define TRANSISTOR_PWM_3_9K_OHMS       1024 /* A Constant DAC voltage to Transistor Input, Transitor partially On (3.9K Ohms between ALDL Connector DIAG Mode Select and Ground) */
#define TRANSISTOR_PWM_SATURATED       4095 /* A Constant DAC voltage to Transistor Input Low Pass Filter, Transitor is Saturated (Short between ALDL Connector DIAG Mode Select and Ground) */

#define SES_ALIVE_COUNT               85000 /* The Service Engine Soon (SES) monitor will send an Error Code 0 (EC0) periodcially as it monitors the Error Codes to keep the Link active */ 
 
#define USER_LED_PIN                     13 /* Adafruit Trinket M0 Red LED */
#define ALDL_160_BAUD_PIN                 0 /* Adafruit Trinket M0 Pin  0: Digtial Input from a 3.3V Zener limiter from the Fiero ALDL Connector 160 Baud Pin (ALDL Pin E) */
#define ALDL_SES_PIN                      2 /* Adafruit Trinket M0 Pin  2: Digital Input from a 3.3V Zener limiter from the Fiero ALDL Connector SES Signal (ALDL Pin D) */
#define ALDL_DIAG_MODE_SELECT_PIN        A0 /* Adafruit Trinket M0 Pin ~1: 12 Bit DAC Output to a Transisitor to control the resistance to the Fiero ALDL Connector Diag Mode Select Pin (ALDL Pin B) */
                                            /* 
                                             *  The Adafruit Trinket M0 has a 12-bit Digital to Analog Converter (DAC) output pin, so there is no need for a simple 
                                             *  Resistor-Capacitor (RC) Low Pass Filter before the Base input of the Transistor.  
                                             */
                       
#endif

#define ERROR_CODE_COARSE_DELAY_COUNTS    5 /* Allows EC0's (NULLs) to be sent over link to keep it active */
#define ERROR_CODE_PULSE_ON_MS          300 /* Light is On for a Count Pulse in milliseconds */
#define ERROR_CODE_PULSE_OFF_MS         200 /* Light is Off for a Count Pulse in milliseconds */
#define ERROR_CODE_PAUSE_MS            1000 /* Light in Off between Digit Counts of an Error Code in milliseconds (accumulates with prior ERROR_CODE_PULSE_OFF_MS to be 1.200 seconds total) */
#define ERROR_CODE_SPACE_MS            2000 /* Light is Off between Error Codes in milliseconds (accumulates with prior ERROR_CODE_PULSE_OFF_MS to be 3.200 seconds total) */
#define ALDL_BIT_DELAY_MS                 6 /* Actual is 6.25 ms for 160 Baud, this is close due to the limted resolution of delay() which is in whole milliseconds */


/* Enumerations */

enum
{
  MODE_GET_ALDL_DATA   = 0,
  MODE_GET_ERROR_CODES = 1
};

enum
{
  DIAG_MODE_NORMAL      = 0,
  DIAG_MODE_DIAGNOSTICS = 1,
  DIAG_MODE_FACTORY     = 2,
  DIAG_MODE_ALDL        = 3
};

enum
{
  SIMULATING_NO  = 0,
  SIMULATING_YES = 1
};


/* Global Variables */

int           Data = 0x00;
int           Mask;
int           Simulating = SIMULATING_NO;
String        Command;
int           Command_Selected = MODE_GET_ALDL_DATA; 
int           Argument;
int           Diag_Mode = DIAG_MODE_NORMAL;
int           Error_Code;
int           Error_Code_Tens;
int           Error_Code_Time_ms;
int           Fine_Delay_ms;
unsigned long Alive_Counter = 0;


/* Function Prototypes */

void Monitor_160_ALDL_Link(void);
void Monitor_SES_Signal(void);
void Simulate_160_ALDL_Link(void);
void Simulate_160_ALDL_Partial_Link(void);
void Simulate_SES_Signal(void); 
int  Determine_Error_Code_Time_In_Millisec(int Error_Code);


/* Functions */

/********************************************************************************
 *
 * There is no main() used in the Ardunio IDE.
 * the Ardunio IDE compiler will call setup() once, then it calls loop() 
 * automatically.
 *
 ********************************************************************************/
void setup() 
{
  /* Set up the USB Console Comms */
  Serial.begin(9600);

  /* Set up Digital Pins */
  pinMode(USER_LED_PIN,      OUTPUT);
  digitalWrite(USER_LED_PIN, 0);
  
  pinMode(ALDL_160_BAUD_PIN, INPUT_PULLUP);
  
  pinMode(ALDL_SES_PIN,      INPUT_PULLUP);

  /* Set the Diag Mode Select Transitor to Normal Mode: Provides an Open to Ground at the Feiro ALDL Diag Mode Select pin */
  analogWrite(ALDL_DIAG_MODE_SELECT_PIN, TRANSISTOR_PWM_OFF);
     
  #if SIMULATE_DEFAULT
  Simulating = SIMULATING_YES;          
  digitalWrite(USER_LED_PIN, 1);
  #endif 
}


/********************************************************************************
 *
 * Monitor or Simulate the Fiero ALDL 160 Baud Data Link or Fiero Error Codes 
 * via the SES Signal (if the Fiero ALDL connector is modified to include the
 * SES signal).
 *
 ********************************************************************************/
void loop() 
{
  /* Initialization */
    
  /* Process */
  
  /* Loop Forever */
  while(1)
  {
    /* Get Command (if any) */
    if(Serial.available() > 0)
    {   
      /* Wait for rest of command to arrive */
      delay(500); 
      Command = Serial.readString();

      /* Parse Command */
      if(Command.compareTo("IDENTIFY\n") == 0)
      {
        Serial.print("Fiero_ALDL_Adapter_V1.6.0\n");
      }   
      else
      if(Command.compareTo("MODE_GET_ALDL_DATA\n") == 0)
      {  
        Command_Selected = MODE_GET_ALDL_DATA;
      }
      else
      if(Command.compareTo("MODE_GET_ERROR_CODES\n") == 0)
      {
        Command_Selected = MODE_GET_ERROR_CODES;
      }
      else
      if(Command.compareTo("SET_DIAGNOSTIC_MODE\n") == 0)
      {       
        delay(100); /* Wait for argument to arrive */
        if(Serial.available() > 0)
        {
          Command = Serial.readString();
          if(Command[0] == '0') Argument = DIAG_MODE_NORMAL;
          if(Command[0] == '1') Argument = DIAG_MODE_DIAGNOSTICS;
          if(Command[0] == '2') Argument = DIAG_MODE_FACTORY;
          if(Command[0] == '3') Argument = DIAG_MODE_ALDL;
           
          switch(Argument)
          {
            default:
            case DIAG_MODE_NORMAL: /* NORM (Open) */
              Diag_Mode = DIAG_MODE_NORMAL;
              /* Set the Diag Mode Select Transitor provide an Open to Ground at the Feiro ALDL Diag Mode Select pin */  
              analogWrite(ALDL_DIAG_MODE_SELECT_PIN, TRANSISTOR_PWM_OFF);
            break;

            case DIAG_MODE_DIAGNOSTICS: /* DMDIAG (0 Ohms) */
              Diag_Mode = DIAG_MODE_DIAGNOSTICS;
              /* Set Diag Mode Select Transitor to provide a Short to Ground at the Feiro ALDL Diag Mode Select pin */  
              analogWrite(ALDL_DIAG_MODE_SELECT_PIN, TRANSISTOR_PWM_SATURATED);
            break;

            case 2: /* DMFACT (3.9K Ohms) */
              Diag_Mode = DIAG_MODE_FACTORY;
              /* Set Diag Mode Select Transitor to provide 3.9K Ohms to Ground at the Feiro ALDL Diag Mode Select pin */  
              analogWrite(ALDL_DIAG_MODE_SELECT_PIN, TRANSISTOR_PWM_3_9K_OHMS);      
            break;

            case 3: /* DMALDL (10K Ohms) */
              Diag_Mode = DIAG_MODE_ALDL;
              /* Set Diag Mode Select Transitor to provide 10K Ohms to Ground at the Feiro ALDL Diag Mode Select pin */  
              analogWrite(ALDL_DIAG_MODE_SELECT_PIN, TRANSISTOR_PWM_10K_OHMS);
            break;
          }
        }
      }
      else
      if(Command.compareTo("GET_DIAGNOSTIC_MODE\n") == 0)
      {
        /* Get Diag Mode of ALDL Link */
        Serial.write('0' + Diag_Mode);
        Serial.print("\n");
      }
      else
      if(Command.compareTo("MODE_SIMULATION\n") == 0)
      {
        /* Enable Simulation */
        Simulating = SIMULATING_YES;
        digitalWrite(USER_LED_PIN, 1);
      }
      else
      if(Command.compareTo("MODE_MONITOR\n") == 0)
      {
        /* Disable Simulation */
        Simulating = SIMULATING_NO;
        digitalWrite(USER_LED_PIN, 0);
      }
    }
    
    /* Process Command */
    switch(Command_Selected)
    {
      default:
      case MODE_GET_ALDL_DATA: 
        if(Simulating == SIMULATING_YES)
        {
          Simulate_160_ALDL_Link();
          //Simulate_160_ALDL_Partial_Link();
        }
        else
        {
          Monitor_160_ALDL_Link();
        }
      break;
  
      case MODE_GET_ERROR_CODES:
        if(Simulating == SIMULATING_YES)
        {
          Simulate_SES_Signal();
        }
        else
        {
          Monitor_SES_Signal();
        }
      break;
    }  
  }

  /* Termination */

  /* loop() is automatically restarted whe compiled using the Arduino IDE */
}


/********************************************************************************
 *
 * 
 *
 ********************************************************************************/
void Monitor_160_ALDL_Link(void) 
{
  /* Initialization */
  
  int           Break_Flag;
  unsigned long Microseconds = 0;
  unsigned long Start_Time = 0;
  unsigned long Pulse_Width = 0;

  /* Process */
      
  Break_Flag = 0;
  while(1)
  {
    /* 
       Each ALDL Bit lasts 6.25ms in duration, a Active Low Pulse for a Bit '0' is a much less than 3.125ms, 
       but an Active Low Pulse for a Bit '1' is a much more than 3.125ms. 
       pusleIn() could not used because a '1' bit's active pulse is over 5ms and this does not leave enough 
       time to send an ASCII '1' byte (Start + 8Data + Stop at 9600 Baud) over the serial port.
       Measuring the pulse using the microsceond timer allows to abort the pulse measurement after it reaches
       midway through the bit period (a '1' is occuring), thus giving plently of time to send the ASCII '1'
       over the serial port tothe PC at 9600 Baud. 
    */

    /* 
      The ALDL Bit could be in either state (Inactive High or Active Low) at this point.
      Toss out first pulse to be sure not within a partial pulse as the stream is entered.
    */

    /*
                    ms
              ------------
                ECM Type
              ------------
                VN     C3
              -----  ----- 
      Start   0.500  0.351
      Data    4.750  1.557
      Sample  2.875  1.130
      Stop    1.000  4.381
      Total   6.250  6.289

                         Sample 
                           |
           .Start.    Data |        .Stop .
           .     .         v        .     .
      VN --|_____|------------------------|
         --|________________________|-----|
                    ^
                    |
                    
                 Sample
                   |
           .Start.Data.        Stop        . 
           .    .  v  .                    .
      C3 --|____|--------------------------|
         --|__________|--------------------|
                    ^
                    |
                  Sample for both VN or C3
                  1.500 ms    
    */

    /* Start by finding an Inactive High to Active Low transition */
    while(digitalRead(ALDL_160_BAUD_PIN) == 1)
    {
      /* Check for Command Change */
      if(Serial.available() > 0) {Break_Flag = 1; break;}
    }
    if(Break_Flag == 1) break;
    while(digitalRead(ALDL_160_BAUD_PIN) == 0)
    {
      /* Check for Command Change */
      if(Serial.available() > 0) {Break_Flag = 1; break;}
    }
    if(Break_Flag == 1) break;
  
    /* The ALDL Bit has returned to Inactive High at this point, ready for first full pulse */
    /* Detect ALDL Bits */
    while(1)
    {
      /* Check for Command Change */
      if(Serial.available() > 0) {Break_Flag = 1; break;}
      
      /* Wait for ALDL Bit to go Active Low */
      while(digitalRead(ALDL_160_BAUD_PIN) == 1)
      {
        /* Check for Command Change */
        if(Serial.available() > 0) {Break_Flag = 1; break;}
      }
      if(Break_Flag == 1) break;
      
      /* Start Timer */
      Start_Time = micros();
      
      /* ALDL Bit just whent Active Low, Timer Started for Pulse Width */ 
      while(1)
      {
        /* Check for Command Change */
        if(Serial.available() > 0) {Break_Flag = 1; break;}
  
        /* Check Timer */
        Microseconds = micros();
        if(Microseconds < Start_Time)
        {
          /* About every 71.583 minutes, micros() will modulated back past 0, adjust for this */
          Microseconds = (0xFFFFFFFF - Start_Time) + Microseconds;
          Start_Time   = 0;
        }
        
        /* Measure Pulse Width */
        Pulse_Width =  Microseconds - Start_Time;
  
        /* Sample Pulse 1.5ms After Falling Edge (Start) of pulse (a compromise between VN and C3 ECMs) */
        if(Pulse_Width > 1500)
        {
          if(digitalRead(ALDL_160_BAUD_PIN) == 1)
          {
            /* Active Low Pulse is Short (returned to 1 early) '0' is detected */
            Serial.print("0");
          }
          else /* ALDL_160_BAUD_PIN == 0 */
          {
            /* Active Low Pulse is Long (returns to 1 later) a '1' is detected */
            Serial.print("1");
          }
          
          /* Wait for pulse to return to Inactive High */
          while(digitalRead(ALDL_160_BAUD_PIN) == 0)
          {
            /* Check for Command Change */
            if(Serial.available() > 0) {Break_Flag = 1; break;}
          }
          if(Break_Flag == 1) break;
          
          /* Get Ready for Next Pulse */
          break;
        }
      }
      if(Break_Flag == 1) break;
    }
    if(Break_Flag == 1) break;
  }
      
  /* Termination */
      
  return;
}


/********************************************************************************
 *
 * 
 *
 ********************************************************************************/
void Monitor_SES_Signal(void) 
{
  /* Initialization */
    
  int           Break_Flag;
  unsigned long Microseconds = 0;
  unsigned long Start_Time = 0;
  unsigned long Pulse_Width = 0;
  
  /* Process */

  Break_Flag = 0;
  while(1)
  {
    /* 
       Could not use pulseIn() because timing of both the Active Low and Inactive High states
       would cause missing the On(Active Low)/Off(Inactive High) edges, which is required by pulseIn().
       Measuring the pulses and timing between them using the microsceond timer allows both
       to be measured accurately without missing edges.
    */
      
    /* 
      The SES signal could be in either state (Inactive High or Active Low) at this point.
      Synchronize to Error Code stream, wait for SES to be Inactive High for at least 1.5 seconds (Between Error Codes)   
    */
      
    /* Start by finding an Inactive High to Active Low transition */
    while(digitalRead(ALDL_SES_PIN) == 1)
    {
      /* Check for Command Change */
      if(Serial.available() > 0) {Break_Flag = 1; break;}

      Alive_Counter++;
      if((Alive_Counter % SES_ALIVE_COUNT) == 0) Serial.write((unsigned char)0); /* Send not possible EC0 to keep Comms Link Active */
    }
    if(Break_Flag == 1) break;
    while(digitalRead(ALDL_SES_PIN) == 0)
    {
      /* Check for Command Change */
      if(Serial.available() > 0) {Break_Flag = 1; break;}

      Alive_Counter++;
      if((Alive_Counter % SES_ALIVE_COUNT) == 0) Serial.write((unsigned char)0); /* Send not possible EC0 to keep Comms Link Active */
    }
    if(Break_Flag == 1) break;
    
    /* SES is Inactive High, Start Timer for the Inactive High Pulse Width */
    Start_Time = micros();
    
    /* Wait for SES to go Active Low */
    while(digitalRead(ALDL_SES_PIN) == 1)
    {
      /* Check for Command Change */
      if(Serial.available() > 0) {Break_Flag = 1; break;}

      Alive_Counter++;
      if((Alive_Counter % SES_ALIVE_COUNT) == 0) Serial.write((unsigned char)0); /* Send not possible EC0 to keep Comms Link Active */
    }
    if(Break_Flag == 1) break;
    
    /*  Stop Timer */ 
    Microseconds = micros();
    if(Microseconds < Start_Time)
    {
      /* About every 71.583 minutes, micros() will modulated back past 0, adjust for this */
      Microseconds = (0xFFFFFFFF - Start_Time) + Microseconds;
      Start_Time   = 0;
    } 
    
    /* Determine Inctive High Pulse Width */
    Pulse_Width = Microseconds - Start_Time;
    
    /* If greater than 1.5 seconds, now Syncronized to the next Error Code Pattern */ 
    if(Pulse_Width > 1500000)
    {
      /* SES is Active Low, Start Timer for the Active Low Pulse Width */
      Error_Code_Tens = 1;
      Error_Code = 0;
      /* In Sync, ready for Error Code Pulse Measurements */
      Start_Time = micros();
      break;
    }
    else
    {
      ; /* Try Next Inactive High Pulse */
    }
  }
  
  /* 
    Measure Error Code pulses. 
    SES just went Active Low after being Inactive High > 1.5 seconds at this point, 
    Pulse Width Timer was started. 
  */
  while(1)
  {
    /* Check for Command Change */
    if(Serial.available() > 0) {Break_Flag = 1; break;}

    Alive_Counter++;
    if((Alive_Counter % SES_ALIVE_COUNT) == 0) Serial.write((unsigned char)0); /* Send not possible EC0 to keep Comms Link Active */
    
    /* Wait for pulse to return to Inactive High */
    while(digitalRead(ALDL_SES_PIN) == 0)
    {
      /* Check for Command Change */
      if(Serial.available() > 0) {Break_Flag = 1; break;}

      Alive_Counter++;
      if((Alive_Counter % SES_ALIVE_COUNT) == 0) Serial.write((unsigned char)0); /* Send not possible EC0 to keep Comms Link Active */
    }
    if(Break_Flag == 1) break;
    
    /* Stop Timer */ 
    Microseconds = micros();
    if(Microseconds < Start_Time)
    {
      /* About every 71.583 minutes, micros() will modulated back past 0, adjust for this */
      Microseconds = (0xFFFFFFFF - Start_Time) + Microseconds;
      Start_Time   = 0;
    } 
    
    /* Determine Active Low Pulse Width */
    Pulse_Width = Microseconds - Start_Time;
    
    /* Act on Active Low Pulse Width */
    if(Pulse_Width < 500000)
    {
      /* If Inactive Time < 0.5 seconds, count the digit */
      if(Error_Code_Tens == 1) Error_Code += 10;
      else                     Error_Code +=  1;
    }
    else
    {
      /* Error - Active Low pulse should never be more than 0.5 seconds */
    }
    
    /* SES is Inactive High, Start Timer for the Inactive High Pulse Width */
    Start_Time = micros();
    
    /* Wait for pulse to go Active Low */
    while(digitalRead(ALDL_SES_PIN) == 1)
    {
      /* Check for Command Change */
      if(Serial.available() > 0) {Break_Flag = 1; break;}

      Alive_Counter++;
      if((Alive_Counter % SES_ALIVE_COUNT) == 0) Serial.write((unsigned char)0); /* Send not possible EC0 to keep Comms Link Active */
    }
    if(Break_Flag == 1) break;
    
    /* Stop Timer */ 
    Microseconds = micros();
    if(Microseconds < Start_Time)
    {
      /* About every 71.583 minutes, micros() will modulated back past 0, adjust for this */
      Microseconds = (0xFFFFFFFF - Start_Time) + Microseconds;
      Start_Time   = 0;
    } 
    
    /* Measure Inactive High Pulse Width */
    Pulse_Width = Microseconds - Start_Time;
     
    /* Act on Inactive High Pulse Width */
    if(Pulse_Width < 500000)
    {
      ; /* Inactive Time < 0.5 seconds, still in Current Digit */
    }
    else
    if(Pulse_Width < 1500000)
    {
      /* Inactive Time > 0.5 and < 1.5 seconds, going to next digit */
      Error_Code_Tens = 0;
    }
    else
    {
      /* Inactive Time > 1.5 seconds (on its way to 2.5 seconds), going to next Error Code */
      /* Send Error Code to PC */
      Serial.write(Error_Code);
      /* Prep for Net Error Code */
      Error_Code_Tens = 1;
      Error_Code = 0;
    }

    /* Wait for pulse to go Active Low */
    while(digitalRead(ALDL_SES_PIN) == 1)
    {
      /* Check for Command Change */
      if(Serial.available() > 0) {Break_Flag = 1; break;}

      Alive_Counter++;
      if((Alive_Counter % SES_ALIVE_COUNT) == 0) Serial.write((unsigned char)0); /* Send not possible EC0 to keep Comms Link Active */
    }
    if(Break_Flag == 1) break;
    
    /* Start Timer for Active Low Pulse Width */
    Start_Time = micros();
  }
  
  /* Termination */
  
  return;
}


/********************************************************************************
 *
 * 
 *
 ********************************************************************************/
void Simulate_160_ALDL_Link(void) 
{
  /* Initialization */
    
  int Break_Flag;

  /* Process */

  Break_Flag = 0;
  while(1)
  {
    /* Sync Byte: 9 Bits (all '1's) */
    for(int Bit = 0; Bit < 9; Bit++)
    {
      /* Check for Command Change */
      if(Serial.available() > 0) {Break_Flag = 1; break;}

      /* Send Bit */
      Serial.print("1");
     
      /* Simulate Bit Timing */
      delay(ALDL_BIT_DELAY_MS);
    }
    if(Break_Flag == 1) break;
            
    /* 24 Incrementing Data Bytes: 9 Bits Each */
    for(int Byte = 0; Byte < 25; Byte++)
    {
      /* Check for Command Change */
      if(Serial.available() > 0) {Break_Flag = 1; break;}
      
      /* Send a Sync Bit to PC ('0' for Data) */
      Serial.print("0");

       /* Simulate Bit Timing */
      delay(ALDL_BIT_DELAY_MS);
            
      /* 8 Data Bits */ 
      Mask = 0x80;
      for(int Bit = 0; Bit < 8; Bit++)
      {
        /* Check for Command Change */
        if(Serial.available() > 0) {Break_Flag = 1; break;}

        /* Send Bit to PC */
        if((Data & Mask) == Mask) Serial.print("1");
        else                      Serial.print("0");
        Mask >>= 1;
        Mask &= 0x7F;
             
        /* Simulate Bit Timing */
        delay(ALDL_BIT_DELAY_MS);
      }   
      if(Break_Flag == 1) break; 

      /* Next Data Pattern */
      Data++;
      if(Data > 0xFF) Data = 0x00;
    }
    if(Break_Flag == 1) break;
  }

  /* Termination */
  
  return;
}


/********************************************************************************
 *
 * 
 *
 ********************************************************************************/
void Simulate_160_ALDL_Partial_Link(void) 
{
  /* Initialization */
    
  int Break_Flag;

  /* Process */

  Break_Flag = 0;
  while(1)
  {
    /* Sync Byte: 9 Bits (all '1's) */
    for(int Bit = 0; Bit < 9; Bit++)
    {
      /* Check for Command Change */
      if(Serial.available() > 0) {Break_Flag = 1; break;}

      /* Send Bit */
      Serial.print("1");
     
      /* Simulate Bit Timing */
      delay(ALDL_BIT_DELAY_MS);
    }
    if(Break_Flag == 1) break;
            
    /* 4 Incrementing Data Bytes: 9 Bits Each */
    for(int Byte = 0; Byte < 4; Byte++)
    {
      /* Check for Command Change */
      if(Serial.available() > 0) {Break_Flag = 1; break;}
      
      /* Send a Sync Bit to PC ('0' for Data) */
      Serial.print("0");

       /* Simulate Bit Timing */
      delay(ALDL_BIT_DELAY_MS);
            
      /* 8 Data Bits */ 
      Mask = 0x80;
      for(int Bit = 0; Bit < 8; Bit++)
      {
        /* Check for Command Change */
        if(Serial.available() > 0) {Break_Flag = 1; break;}

        /* Send Bit to PC */
        if((Data & Mask) == Mask) Serial.print("1");
        else                      Serial.print("0");
        Mask >>= 1;
        Mask &= 0x7F;
             
        /* Simulate Bit Timing */
        delay(ALDL_BIT_DELAY_MS);
      }   
      if(Break_Flag == 1) break; 

      /* Next Data Pattern */
      Data++;
      if(Data > 0xFF) Data = 0x00;
    }
    if(Break_Flag == 1) break;
  }

  /* Termination */
  
  return;
}

/********************************************************************************
 *
 * 
 *
 ********************************************************************************/
void Simulate_SES_Signal(void) 
{
  /* Initialization */
    
  int Break_Flag;

  /* Process */

  Break_Flag = 0;
  while(1)
  {  
    /* Emulate 3 EC12's */
    Error_Code = 12;
    Error_Code_Time_ms = Determine_Error_Code_Time_In_Millisec(Error_Code);
    Fine_Delay_ms = (int)(Error_Code_Time_ms / ERROR_CODE_COARSE_DELAY_COUNTS);
    for(int Repeat = 0; Repeat < 3; Repeat++)
    {
      for(int Coarse_Delay = 0; Coarse_Delay < ERROR_CODE_COARSE_DELAY_COUNTS; Coarse_Delay++)
      {
        /* Check for Command Change */
        if(Serial.available() > 0) {Break_Flag = 1; break;}

        /* Send not possible EC0 to keep Comms Link Active */
        Serial.write((unsigned char)0); 
        
        /* Simulate Error Code Timing */
        delay(Fine_Delay_ms);
      } 
      if(Break_Flag == 1) break;
      
      /* Send the Error Code */
      Serial.write(Error_Code); 
      
      /* Simulate Error Code Timing */
      delay(ERROR_CODE_SPACE_MS);
    }
    if(Break_Flag == 1) break;

    /* Emulate 3 EC13's */
    Error_Code = 13;
    Error_Code_Time_ms = Determine_Error_Code_Time_In_Millisec(Error_Code);
    Fine_Delay_ms = (int)(Error_Code_Time_ms / ERROR_CODE_COARSE_DELAY_COUNTS);
    for(int Repeat = 0; Repeat < 3; Repeat++)
    {
      for(int Coarse_Delay = 0; Coarse_Delay < ERROR_CODE_COARSE_DELAY_COUNTS; Coarse_Delay++)
      {
        /* Check for Command Change */
        if(Serial.available() > 0) {Break_Flag = 1; break;}

        /* Send not possible EC0 to keep Comms Link Active */
        Serial.write((unsigned char)0); 

        /* Simulate Error Code Timing */
        delay(Fine_Delay_ms);
      }
      if(Break_Flag == 1) break;
      
      /* Send the Error Code */
      Serial.write(Error_Code); 
      
      /* Simulate Error Code Timing */
      delay(ERROR_CODE_SPACE_MS);
    }
    if(Break_Flag == 1) break;
    
    /* Emulate 3 EC61's */

    Error_Code = 61;
    Error_Code_Time_ms = Determine_Error_Code_Time_In_Millisec(Error_Code);
    Fine_Delay_ms = (int)(Error_Code_Time_ms / ERROR_CODE_COARSE_DELAY_COUNTS);
    for(int Repeat = 0; Repeat < 3; Repeat++)
    {
      for(int Coarse_Delay = 0; Coarse_Delay < ERROR_CODE_COARSE_DELAY_COUNTS; Coarse_Delay++)
      {
        /* Check for Command Change */
        if(Serial.available() > 0) {Break_Flag = 1; break;}

        /* Send not possible EC0 to keep Comms Link Active */
        Serial.write((unsigned char)0); 

        /* Simulate Error Code Timing */
        delay(Fine_Delay_ms);
      }
      if(Break_Flag == 1) break;
      
      /* Send the Error Code */
      Serial.write(Error_Code);
      
      /* Simulate Error Code Timing */
      delay(ERROR_CODE_SPACE_MS);
    }
    if(Break_Flag == 1) break;
  }
  
  /* Termination */
  
  return;
} 
 


/********************************************************************************
 *
 * Error Code 99 may take up to 13.0 seconds but the highest expected Error Code 
 * for a Fiero is 55 which takes about 11.5 seconds. EC12 takes 6.0 seconds.
 *
 ********************************************************************************/
int Determine_Error_Code_Time_In_Millisec(int Error_Code)
{
  /* Initialization */
    
  unsigned long Time_ms;
  unsigned int  Tens;
  unsigned int  Ones;
  unsigned long Pause;

  /* Process */
  
  if(Error_Code > 0)
  {
    Tens = Error_Code / 10;
    Ones = Error_Code - (Tens * 10);

    if(Tens > 0) Pause = ERROR_CODE_PAUSE_MS;
    else         Pause = 0;
   
    Time_ms = (Tens * (ERROR_CODE_PULSE_ON_MS + ERROR_CODE_PULSE_OFF_MS)) + 
               Pause + 
              (Ones * (ERROR_CODE_PULSE_ON_MS + ERROR_CODE_PULSE_OFF_MS)) +
               Pause;
  }
  else
  {
    Time_ms = 0;
  }

  /* Termination */
  
  return Time_ms;
}

/* End */

